import asyncio
import traceback
from typing import Optional
import typer

from voice_synthesis.audio.audio_file_writer import AudioFileWriter
from voice_synthesis.audio.audio_player import AudioPCMPlayer
from voice_synthesis.audio.audio_utils       import AudioFormat, base64_to_bytes
from voice_synthesis.core                    import constants
from voice_synthesis.core.websocket_client   import WebSocketClient
from voice_synthesis.core.logger             import Logger
from voice_synthesis.api.vdkclient           import VdkClient

app = typer.Typer()

@app.command()
def start(
    scheme:     str  = typer.Option(constants.Defaults.DEFAULT_VDK_SCHEME, "--scheme", "-P", help="Protocol to use for the VDK service (http or https)."),
    host:       str  = typer.Option(constants.Defaults.DEFAULT_VDK_HOST,   "--host",   "-h", help="VDK service host."),
    port:       int  = typer.Option(constants.Defaults.DEFAULT_VDK_PORT,   "--port",   "-p", help="VDK service port."),
    text:       str  = typer.Option(None,  "--text", "-t",  help="Text to synthesize."),
    voice:      str  = typer.Option(None,  "--voice", "-v", help="Voice to use."),
    file_name:  str  = typer.Option(None,  "--file", "-f",  help="If given, will output to a file instead of the speaker."),
    listVoices: bool = typer.Option(False, "--list", "-l",  help="List available voices."),
    verbose:    bool = typer.Option(False, "--verbose"),
):
    code = asyncio.run(run(scheme=scheme, host=host, port=port, text=text, voice=voice, file_name=file_name, listVoices=listVoices, verbose=verbose))
    typer.Exit(code)

async def run(
    scheme: str,
    host: str,
    port: int,
    text: str,
    voice: str,
    file_name: str,
    listVoices: bool,
    verbose: bool

):
    # -- validating arguments --
    if listVoices:
        pass
    elif not text:
        Logger.error("You must specify a text.")
        return 1

    # -- Logging configuration --
    Logger.verbose = verbose
    Logger.debug(" -- configuration -- ")
    Logger.debug(f" scheme:                 {scheme}")
    Logger.debug(f" host:                   {host}")
    Logger.debug(f" port:                   {port}")
    Logger.debug(f" text:                   {text}")
    Logger.debug(f" voice:                  {voice}")
    Logger.debug(f" file_name:              {file_name}")
    Logger.debug(f" listVoices:             {listVoices}")
    Logger.debug(f" verbose:                {verbose}")
    Logger.debug(" ------------------------ ")

    # -----------------------------------------------------------------
    # ---------------------- running application ----------------------
    # -----------------------------------------------------------------

    client = VdkClient(scheme, host, port)
    wsClient = None
    try:
        # ------- Making sure service is reachable and ready. -------
        if not await client.health.check_healthz():
            Logger.error("❌ The VDK service is not ready.")
            return 1

        Logger.info(f"✅ VDK service is reachable.")

        voices = await client.voice_synthesis.get_available_voices()

        # ------- Did the user requested a list of voices ? -------
        if listVoices:
            Logger.info(f"Available voices: {voices}")
            return 0

        if voice and voice not in voices:
            Logger.error(f"❌ Voice {voice} is not available.")
            return 1

        if len(voices) == 0:
            Logger.error("❌ No voice available.")
            return 1

        selectedVoice = voice if voice else voices[0]

        # ------- Asking for a synthesis (retrieving a token) -------
        token = await client.voice_synthesis.synthesize(text, selectedVoice)
        if token is None:
            Logger.error("❌ Could not retrieve a token.")
            return 1

        Logger.info(f"✅ Token: {token}")

        # ------- Starting the websocket connection -------
        format = AudioFormat(22050, 1, 16)
        file = None
        # check wav
        if file_name and file_name.endswith(".wav"):
            file = AudioFileWriter(file_name, format)
        # check pcm
        elif file_name:
            file = AudioFileWriter(file_name)

        player = AudioPCMPlayer() if not file else None
        wsClient = WebSocketClient()
        wsClient.on_error = lambda ex: Logger.error("❌ WebSocket error: " + str(ex))
        wsClient.on_close = lambda: Logger.debug("WebSocket connection closed.")
        wsClient.on_message = lambda data: handle_message(data, file, player)
        await wsClient.connect(client.ws_uri(token))

        Logger.info("✅ WebSocket connection established.")

        await wsClient.wait_for_completion()
        if file: file.close()
        elif player: player.wait_for_completion()

        Logger.info("✅ Done.")

    except Exception as e:
        Logger.error("❌ Application failed: " + "\n".join(traceback.format_exception(type(e), e, e.__traceback__)))
        return 1
    finally:
        if wsClient is not None:
            await wsClient.close()
        await client.close()


    return 0

def handle_message(data, file: Optional[AudioFileWriter] = None, player: Optional[AudioPCMPlayer] = None):
    logMessage(data)
    if   "event" in data: Logger.info(f"Event: {data['event']['code_string']}")
    elif "error" in data: Logger.error(f"Error: {data['error']['code']}")
    elif "data"  in data and "last" in data:
        decodedAudio = base64_to_bytes(data["data"], len(constants.B64_PCM_PREFIX))
        if file: file.write_chunk(decodedAudio)
        elif player: player.add_samples(decodedAudio)

def logMessage(data):
    display = dict(data)
    if "data" in display:
        display["data"] = "<base64data>"

    Logger.debug(f"Socket message: {display}")

def main():
    app()
