﻿using System.Collections.Generic;
using System.IO;
using System.Threading.Tasks;
using System.Threading;
using System;
using VdkServiceClient.Api;
using VdkServiceClient.Audio;
using VdkServiceClient.Utils;
using VdkServiceClient.WebSocket;
using System.Json;
using CommandLine;

namespace VoiceSynthesis
{
    class Program
    {
        public static void Main(string[] args)
        {
            CommandLine.Parser.Default.ParseArguments<Options>(args)
                .WithParsed(opts =>
                {
                    if (!opts.Validate(out var err))
                    {
                        Logger.Error($"❌ {err}");
                        return;
                    }

                    Logger.SetVerbose(opts.Verbose);
                    Logger.Debug("Verbose mode enabled.");

                    RunApplication(opts).Wait();
                });
        }

        private static async Task RunApplication(Options opts)
        {
            var vdkClient = new VdkClient(opts.Scheme, opts.Host, opts.Port);

            // ------- Making sure service is reachable and ready. -------
            if (!await vdkClient.Core.Healthz())
            {
                Logger.Error("❌ VDK service is not reachable.");
                return;
            }
            Logger.Info("✅ VDK service is reachable.");

            // ------- Display a list of availables models and make sure model/slot(optional) exists -------
            var availableVoices = new List<(string voiceId, int sampleRate)>();
            if (!await vdkClient.VoiceSynthesis.GetAvailableVoicesAsync(availableVoices) || availableVoices.Count == 0)
            {
                Logger.Error("❌ Could not retrieve available voices. (Is the service running with synthesis configuration ?)");
                return;
            }

            // Did the user requested to list available voices?
            if (opts.ListVoices)
            {
                Logger.Info("Available voices:");
                foreach (var voice in availableVoices)
                {
                    Logger.Info($"- {voice}");
                }
                return; 
            }

            // ------- Asking for a synthesis (retrieving a token) -------
            if (opts.Voice != null && !availableVoices.Exists(x => x.voiceId == opts.Voice))
            {
                Logger.Error($"❌ Voice {opts.Voice} is not available.");
                return;
            }

            var selectedVoiceId = opts.Voice ?? availableVoices[0].voiceId;
            var sampleRate = opts.Voice != null ? availableVoices.Find(x => x.voiceId == selectedVoiceId).sampleRate : availableVoices[0].sampleRate;
            var token = await vdkClient.VoiceSynthesis.SynthesizeAsync(opts.Text, selectedVoiceId);
            
            if (string.IsNullOrWhiteSpace(token))
            {
                Logger.Error("❌ Failed to get token from VdkService.");
                return;
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            bool isWav = !string.IsNullOrEmpty(opts.Output) && Path.GetExtension(opts.Output).ToLower() == ".wav";
            var cancellationTokenSrc = new CancellationTokenSource();
            var audioPlayer = string.IsNullOrEmpty(opts.Output) ? new AudioPlayer(sampleRate) : null;
            var outFile = !string.IsNullOrEmpty(opts.Output) && isWav ? new AudioFileWriter(opts.Output, 22050, 1) : null;
            var ws = new WebSocketClient();
            ws.OnClose += () => {
                Logger.Debug("WebSocket closed.");
                cancellationTokenSrc.Cancel();
            };
            ws.OnError += ex => {
                Logger.Error($"WebSocket error: {ex.Message}");
                cancellationTokenSrc.Cancel();
            };
            ws.OnMessage += json => {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cancellationTokenSrc.Cancel();
                }
                else
                {
                    HandleAudioData(json, audioPlayer, outFile, cancellationTokenSrc); // see below for HandleResult
                }
            };

            await ws.ConnectAsync(vdkClient.wsUrl(token).ToString(), null, cancellationTokenSrc.Token);
            if (!ws.IsConnected)
            {
                Logger.Error("❌ WebSocket connection failed.");
                return;
            }
            Logger.Info("✅ WebSocket connected.");

            // Wait until last audio packet is received
            cancellationTokenSrc.Token.WaitHandle.WaitOne();
            if (audioPlayer != null)
            {
                Logger.Debug("Playback completing...");
                audioPlayer.WaitForCompletion();
            }
            outFile?.Dispose();
            ws?.Dispose();
            audioPlayer?.Dispose();
            Logger.Info("✅ Done.");
        } // !RunAsync

        private static void HandleAudioData(JsonValue audio, AudioPlayer audioPlayer, AudioFileWriter outFile, CancellationTokenSource cts)
        {
            try
            {
                // Audio data must contains 'data' and 'last' fields
                if (!audio.ContainsKey("data") || !audio.ContainsKey("last"))
                {
                    return;
                }

                var buffer = AudioBuffer.FromAudioPacket(audio, audioPlayer.SampleRate, audioPlayer.Channels);
                if (buffer.Data.Length > 0)
                {
                    outFile?.WriteChunk(buffer.Data);      // Writing audio in output.pcm
                    audioPlayer?.AddAudioBuffer(buffer);  // Playing audio
                }

                if (audio["last"])
                {
                    
                    Logger.Info("✅ Received last chunk.");
                    cts.Cancel();
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Failed to parse Synthesize result: {ex.GetType().Name}: {ex.Message}\n{ex.StackTrace}");
            }
        }
    }
}
