using System;
using System.IO;
using System.Json;
using System.Net;
using System.Net.Http;
using System.Runtime.CompilerServices;
using VdkServiceClient.Utils;

namespace VdkServiceClient.Http
{
    public class HttpResult
    {
        public HttpStatusCode StatusCode { get; private set; }
        public JsonValue Data             { get; private set; } = new JsonObject();
        public string RawContent         { get; private set; } = string.Empty;
        public bool IsSuccess            { get; private set; }
        public string ErrorMessage      { get; private set; }

        public static HttpResult FromSuccess(HttpResponseMessage response, string raw)
        {
            var result = new HttpResult();
            result.StatusCode = response.StatusCode;
            result.RawContent = raw;
            result.Data = JsonValue.Parse(raw) ?? new JsonObject();
            result.IsSuccess = true;
            return result;
        }

        public static HttpResult FromError(HttpResponseMessage response, string raw)
        {
            var result = new HttpResult();
            result.StatusCode = response.StatusCode;
            result.RawContent = raw;
            result.Data = JsonValue.Parse(raw) ?? new JsonObject();
            result.IsSuccess = false;
            result.ErrorMessage = $"HTTP {(int)response.StatusCode}: {response.ReasonPhrase}";
            return result;
        }

        public static HttpResult FromException(
                Exception ex,
                [CallerFilePath] string file = "",
                [CallerLineNumber] int line = 0,
                [CallerMemberName] string member = "")
        {
            var result = new HttpResult();
            result.StatusCode = HttpStatusCode.InternalServerError;
            result.IsSuccess = false;
            result.ErrorMessage = $"[{Path.GetFileName(file)}:{line} → {member}] {ex.GetType().Name}: {ex.Message}";
            return result;
        }

        public void displayErrors()
        {
            if (Data?["errors"] is JsonArray errors)
            {
                foreach (var errNode in errors)
                {
                    var file = errNode?["file_name"] ?? "";
                    var func = errNode?["function_name"] ?? "";
                    int line = errNode?["line_number"] ?? -1;
                    var msg  = errNode?["message"] ?? "";

                    Logger.Error($"Error: {msg} | File: {file} | Function: {func} | Line: {line}");
                }
            }
        }
    }
}