using System.Collections.Generic;
using System.Json;
using System.Threading.Tasks;
using VdkServiceClient.Utils;
using VdkServiceClient.Http;


namespace VdkServiceClient.Api
{
    /// <summary>
    /// Speech Enhancement API.
    /// </summary>
    /// <remarks>
    /// Contains : get available enhancers, enhance (audio), AEC is done through the socket (is_reference)
    /// </remarks>
    /// <param name="http"> A preconfigured http client </param>
    public class SpeechEnhancementApi
    {
        private readonly HttpService _http;

        public SpeechEnhancementApi(HttpService http)
        {
            _http = http;
        }

        /// <summary>
        /// Retrieves the list of available enhancers.
        /// </summary>
        /// <param name="availableEnhancers">The list of available enhancers to fill</param>
        public async Task<bool> GetAvailableEnhancersAsync(List<string> availableEnhancers)
        {
            var result = await _http.GetAsync(Constants.Endpoints.Enhancement.GetEnhancers);
            if (!result.IsSuccess)
                return false;

            if (result.Data is JsonValue root)
            {
                var enhancersNode = root["enhancers"];
                if (enhancersNode is JsonArray enhancers)
                {
                    Logger.Debug("Available enhancers:");
                    foreach (var enhancer in enhancers)
                    {
                        availableEnhancers.Add(enhancer);
                        Logger.Debug($"- {enhancer}");
                    }
                }
                return true;
            }

            Logger.Error($"❌ {result.StatusCode}: {result.ErrorMessage}");
            return false;
        } // !GetAvailableEnhancersAsync

        /// <summary>
        /// Make an enhancement request.
        /// </summary>
        /// <param name="enhancerName">The name of the enhancer</param>
        public async Task<string> EnhanceAsync(string enhancerName)
        {
            var json = new JsonObject
            {
                ["speech_enhancer"] = enhancerName
            };

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Enhancement.Enhance, json);

            if (result.IsSuccess)
            {
                if (result.Data is JsonValue root)
                {
                    return root["token"] ?? "";
                }
            }
            Logger.Error($"❌ {result.StatusCode}: {result.ErrorMessage}");
            return "";
        } // !EnhanceAsync
    } // !SpeechEnhancementApi
}
