from typing  import Optional
from aiohttp import ClientResponse
from http    import HTTPStatus
import json
import os
import traceback

class HttpResult:
    def __init__(self,
                 status_code: HTTPStatus,
                 data: dict,
                 raw: str,
                 is_success: bool,
                 error_message: Optional[str] = None):
        self.status_code = status_code
        self.data = data
        self.raw_content = raw
        self.is_success = is_success
        self.error_message = error_message

    @staticmethod
    def from_success(response : ClientResponse, raw: str):
        try:
            data = json.loads(raw)
        except Exception:
            data = {}

        return HttpResult(
            status_code=HTTPStatus(response.status),
            data=data,
            raw=raw,
            is_success=True
        )

    @staticmethod
    def from_error(response : ClientResponse, raw: str):
        try:
            data = json.loads(raw)
        except Exception:
            data = {}

        return HttpResult(
            status_code=HTTPStatus(response.status),
            data=data,
            raw=raw,
            is_success=False,
            error_message=f"HTTP {response.status}: {response.reason}"
        )

    @staticmethod
    def from_exception(ex: Exception):
        tb = traceback.extract_tb(ex.__traceback__)
        file = os.path.basename(tb[-1].filename) if tb else "?"
        line = tb[-1].lineno if tb else -1
        func = tb[-1].name if tb else "?"

        return HttpResult(
            status_code=HTTPStatus.INTERNAL_SERVER_ERROR,
            data={},
            raw="Exception raised",
            is_success=False,
            error_message=f"[{file}:{line} → {func}] {ex.__class__.__name__}: {ex}"
        )

    def display_errors(self, logger=print):
        errors = self.data.get("errors")
        if not isinstance(errors, list):
            return

        for err in errors:
            file = err.get("file_name")
            func = err.get("function_name")
            line = err.get("line_number")
            msg = err.get("message")

            logger(f"Error: {msg} | File: {file} | Function: {func} | Line: {line}")
