import asyncio
import base64

class AudioFormat:
    def __init__(self, sample_rate: int, channels: int, bits_per_sample: int):
        self.sample_rate = sample_rate
        self.channels = channels
        self.bits_per_sample = bits_per_sample

from voice_recognition.audio.audio_reader import AudioReader
from voice_recognition.audio.audio_streamer import AudioStreamer
from voice_recognition.core import constants
from voice_recognition.core.logger import Logger
from voice_recognition.core.websocket_client import WebSocketClient
from voice_recognition.audio.audio_capture import AudioCapture

def base64_to_bytes(base64_str: str, prefix_length: int) -> bytes:
    return base64.b64decode(base64_str[prefix_length:])

async def stream_microphone(ws: WebSocketClient):
    Logger.info("Using microphone")
    try:
        with AudioCapture(constants.Defaults.DEFAULT_AUDIO_FORMAT) as capture:
            capture.on_audio_data = ws.send_audio
            Logger.info("Capture started.")
            Logger.info("Press Ctrl+C to stop the capture.")
            await ws.wait_for_completion()

    except (KeyboardInterrupt, asyncio.CancelledError):
        Logger.info("Capture stopped by user.")
        await ws.send_audio(b"", True)
    except Exception as e:
        Logger.error(f"❌ Capture failed: {e}")

async def stream_file(ws, file_path: str, format: AudioFormat = None, realtime: bool = True, is_reference: bool = False):
    Logger.info("Using file")

    with AudioReader(file_path, format) as reader:
        pcm        = reader.read_all_bytes()
        format     = reader.format

        async def callback(chunk, is_last):
            await ws.send_audio(chunk, is_last, {"is_reference": is_reference})

        try:
            if realtime: await AudioStreamer.stream_realtime(pcm, format, callback)
            else:        await AudioStreamer.stream(pcm, callback)
        except Exception as e: Logger.error(f"Send failed: {e}")

async def stream_bytes(ws, pcm: bytes, format: AudioFormat, realtime: bool):
    async def callback(chunk, is_last):
        await ws.send_audio(chunk, is_last)

    try:
        if realtime: await AudioStreamer.stream_realtime(pcm, format, callback)
        else:        await AudioStreamer.stream(pcm, callback)
    except Exception as e: Logger.error(f"Send failed: {e}")

    if realtime:
        try:
            await AudioStreamer.stream_realtime(pcm, format, callback)
        except Exception as e:
            Logger.error(f"Send failed: {e}")
