import asyncio
import traceback
import typer

from voice_recognition.audio.audio_utils       import stream_file, stream_microphone
from voice_recognition.core                    import constants, structures
from voice_recognition.core.websocket_client   import WebSocketClient
from voice_recognition.core.logger             import Logger
from pathlib                                 import Path
from voice_recognition.api.vdkclient           import VdkClient

app = typer.Typer()

@app.command()
def start(
    scheme:               str  = typer.Option(constants.Defaults.DEFAULT_VDK_SCHEME, "--scheme", "-P", help="Protocol to use for the VDK service (http or https)."),
    host:                 str  = typer.Option(constants.Defaults.DEFAULT_VDK_HOST,   "--host",   "-h", help="VDK service host."),
    port:                 int  = typer.Option(constants.Defaults.DEFAULT_VDK_PORT,   "--port",   "-p", help="VDK service port."),
    model:                str  = typer.Option(None,  "--model",                "-m", help="The Voice Recognition model to use."),
    slot:                 str  = typer.Option(None,  "--slot",                 "-s", help="Format: slot:value1;value2;value3"),
    stop_at_first_result: bool = typer.Option(False, "--stop-at-first-result", "-r", help="Model is not installed again after first result."),
    file:                 str  = typer.Option(None,  "--file",                 "-f", help="If given, will use a file instead of the microphone."),
    listModels:           bool = typer.Option(False, "--list-models",          "-l", help="List available models."),
    verbose:              bool = typer.Option(False, "--verbose", "-v"),
):
    code = asyncio.run(run(scheme=scheme, host=host, port=port, model=model, slot=slot, stop_at_first_result=stop_at_first_result, file=file, listModels=listModels, verbose=verbose))
    typer.Exit(code)

async def run(
    scheme: str,
    host: str,
    port: int,
    model: str,
    slot: str,
    stop_at_first_result: bool,
    file: str,
    listModels: bool,
    verbose: bool
):
    # -- validating arguments --
    if listModels:
        pass
        pass
    elif not model:
        Logger.error("You must specify a model.")
        return 1
    elif file and (not Path(file).exists() or not Path(file).is_file()):
        Logger.error(f"File {file} does not exist.")
        return 1

    # Format: slot:value1;value2;value3
    slotName = None
    slotValues = None
    if slot:
        if ":" not in slot or len(slot.split(":")) != 2:
            Logger.error("You must specify a slot.")
        else:
            slotName = slot.split(":")[0]
            slotValues = slot.split(":")[1].split(";")

    # -- Logging configuration --
    Logger.verbose = verbose
    Logger.debug(" -- configuration -- ")
    Logger.debug(f" scheme:                 {scheme}")
    Logger.debug(f" host:                   {host}")
    Logger.debug(f" port:                   {port}")
    Logger.debug(f" model:                  {model}")
    Logger.debug(f" slot:                   {slot}")
    Logger.debug(f" stop_at_first_result:   {stop_at_first_result}")
    Logger.debug(f" file:                   {file}")
    Logger.debug(f" listModels:             {listModels}")
    Logger.debug(f" verbose:                {verbose}")
    Logger.debug(" ------------------------ ")

    # -----------------------------------------------------------------
    # ---------------------- running application ----------------------
    # -----------------------------------------------------------------

    client = VdkClient(scheme, host, port)
    wsClient = None
    try:
        # ------- Making sure service is reachable and ready. -------
        if not await client.health.check_healthz():
            Logger.error("❌ The VDK service is not ready.")
            return 1

        Logger.info(f"✅ VDK service is reachable.")

        # ------- Did the user requested a list of models ? -------
        if listModels:
            models = await client.voice_recognition.get_available_models()
            Logger.info(f"Available models: {models}")
            return 0

        # ------- Display a list of availables models and make sure model/slot(optional) exists -------
        if not await client.voice_recognition.check_model(model, slotName):
            if slot is None:
                Logger.error(f"❌ Model {model} does not exist.")
            else:
                Logger.error(f"❌ Model {model} or slot {slotName} does not exist.")
            return 1

        Logger.info(f"✅ Model and slot exists.")

        # ------- Asking for a recognize/enroll (retrieving a token) -------
        token = await client.voice_recognition.recognize(model, slotName, slotValues, stop_at_first_result)
        if token is None:
            Logger.error("❌ Could not retrieve a token.")
            return 1

        Logger.info(f"✅ Token: {token}")

        # ------- Starting the websocket connection -------
        wsClient = WebSocketClient()
        wsClient.on_error = lambda ex: Logger.error("❌ WebSocket error: " + str(ex))
        wsClient.on_close = lambda: Logger.debug("WebSocket connection closed.")
        wsClient.on_message = lambda data: handle_message(data)
        await wsClient.connect(client.ws_uri(token))

        Logger.info("✅ WebSocket connection established.")

        # ------- Start streaming audio -------
        if file:
            await stream_file(wsClient, file)
        else:
            await stream_microphone(wsClient)

        await wsClient.wait_for_completion()
        Logger.info("✅ Done.")

    except Exception as e:
        Logger.error("❌ Application failed: " + "\n".join(traceback.format_exception(type(e), e, e.__traceback__)))
        return 1
    finally:
        if wsClient is not None:
            await wsClient.close()
        await client.close()

    return 0

def handle_message(data):
    Logger.debug(f"Socket message: {data}")
    if "event" in data:    Logger.info(f"Event: {data['event']['code_string']}")
    elif "error" in data:  Logger.error(f"Error: {data['error']['code']}")
    elif "result" in data: handle_recognize_result(data["result"])

def handle_recognize_result(message: dict):
    try:
        result = structures.AsrResult.model_validate(message)
        for hypo in result.hypotheses or []:
            words = []
            for it in hypo.items or []:
                retrieve_words(words, it)

            sentence = " ".join(words)
            Logger.info(f"Hypothesis: {sentence} (Confidence: {hypo.confidence})")

    except Exception as e:
        Logger.error(f"❌ Error decoding recognize result: {e}")
        raise

def retrieve_words(dst: list[str], item: structures.Item):
    if item is None:
        return

    if item.type == "tag" and item.items:
        for child in item.items:
            retrieve_words(dst, child)
    else:
        dst.append(item.orthography or "<null>")

def main():
    app()
