from typing import List, Optional

from voice_recognition.core.constants import Uris
from voice_recognition.core.http_result import HttpResult
from voice_recognition.core.http_service import HttpService
from voice_recognition.core.logger import Logger

class VoiceRecognitionClient:
    def __init__(self, http_service: HttpService):
        self._http: HttpService = http_service

    async def get_available_models(self) -> List[str]:
        available_models = []

        result : HttpResult = await self._http.get(Uris.VoiceRecognition.GET_MODELS)
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return False

        Logger.debug(f"received : {result.raw_content}")

        for key in result.data.keys():
            available_models.append(key)

        return available_models

    async def check_model(self, model: str, slot: str) -> bool:
        result : HttpResult = await self._http.get(Uris.VoiceRecognition.GET_MODELS)
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return False

        Logger.debug(f"received : {result.raw_content}")

        if model not in result.data.keys():
            Logger.error(f"❌ Model {model} not found")
            return False

        # make sure slots exists
        if slot and ("slots" not in result.data[model] or slot not in result.data[model]["slots"]):
            Logger.error(f"❌ Slot {slot} not found")
            return False

        return True

    async def recognize(self,
                        model: str,
                        slotName: Optional[str],
                        slotValues: Optional[List[str]],
                        stop_at_first_result: bool) -> Optional[str]:
        json_body = {}
        if slotName and slotValues:
            json_body = {
                "models": {
                    model: {
                        "slots": {
                            slotName: {
                                "values": slotValues
                            }
                        }
                    }
                },
                "stop_at_first_result": stop_at_first_result
            }
        else:
            json_body = {
                "models": {
                    model: {}
                },
                "stop_at_first_result": stop_at_first_result
            }

        result: HttpResult = await self._http.post(Uris.VoiceRecognition.RECOGNIZE, json_body)

        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        if "token" not in result.data:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        return result.data["token"]
