﻿using System.Collections.Generic;
using System.Threading.Tasks;
using System.Threading;
using VdkServiceClient.Api;
using VdkServiceClient.Audio;
using VdkServiceClient.Utils;
using VdkServiceClient.WebSocket;
using CommandLine;
using System;
using System.Linq;
using System.Json;

namespace VoiceRecognition
{
    class Program
    {
        public static void Main(string[] args)
        {
            CommandLine.Parser.Default.ParseArguments<Options>(args)
                .WithParsed(opts =>
                {
                    if (!opts.Validate(out var err))
                    {
                        Logger.Error($"❌ {err}");
                        return;
                    }

                    Logger.SetVerbose(opts.Verbose);
                    Logger.Debug("Verbose mode enabled.");

                    RunApplication(opts).Wait();
                });
        }

        private static async Task RunApplication(Options opts)
        {
            var vdkClient = new VdkClient(opts.Scheme, opts.Host, opts.Port);

            // ------- Making sure service is reachable and ready. -------
            if (!await vdkClient.Core.Healthz())
            {
                Logger.Error("❌ VDK service is not reachable.");
                return;
            }
            Logger.Info("✅ VDK service is reachable.");

            // ------- Display a list of availables models and make sure model/slot(optional) exists -------
            var availableModels = new List<(string modelName, List<string> slots)>();
            if (!await vdkClient.VoiceRecognition.GetAvailableModelsAsync(availableModels))
            {
                return;
            }

            // Did the user requested to list available models?
            if (opts.ListModels)
            {
                Logger.Info("Available models:");
                foreach (var model in availableModels)
                {
                    Logger.Info($"- {model.modelName}");
                }
                return;
            }

            // ------- Asking for a recognize (retrieving a token) -------
            var token = await vdkClient.VoiceRecognition.RecognizeAsync(opts.Model, opts.Slot, opts.StopAtFirstResult);
            if (string.IsNullOrWhiteSpace(token))
            {
                Logger.Error("❌ Token is null.");
                return;
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            // Setting callbacks too. See HandleRecognizeResult below for handling of results.
            var cts = new CancellationTokenSource();
            var ws = new WebSocketClient();
            ws.OnClose += () =>
            {
                Logger.Debug("WebSocket closed.");
                cts.Cancel();
            };
            ws.OnError += ex =>
            {
                Logger.Error($"WebSocket error: {ex.Message}");
                cts.Cancel();
            };
            ws.OnMessage += json =>
            {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json["event"]["code_string"]}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cts.Cancel();
                }
                else
                {
                    HandleResult(json["result"]);
                }
            };

            await ws.ConnectAsync(vdkClient.wsUrl(token).ToString(), null, cts.Token);
            if (!ws.IsConnected)
            {
                Logger.Error("❌ WebSocket connection failed.");
                return;
            }
            Logger.Info("✅ WebSocket connected.");

            // ------- Streaming audio -------
            if (string.IsNullOrEmpty(opts.InputFile))
            {
                await AudioStreamer.StreamMicrophone(ws, 16000, 1);
            }
            else
            {
                AudioStreamer.StreamFile(ws, opts.InputFile, 16000, 1);
            }

            //cts.Token.WaitHandle.WaitOne();
            Logger.Info("✅ Done");
        }

        static void HandleResult(JsonValue result)
        {
            try
            {
                // See Core/AsrResult.cs for structures.
                var asrResult = result.ToDynamicObject();
                if (asrResult is null)
                {
                    return;
                }

                Logger.Info("Result received (number of hypotheses: " + (asrResult.hypotheses?.Count ?? 0) + ")");

                // Going through hypotheses (The first one is always the best one)
                foreach (var hypothesis in asrResult.hypotheses)
                {
                    Logger.Info($" - Hypothesis: {GetHypothesisText(hypothesis)} (Confidence: {hypothesis.confidence})");
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Failed to parse ASR result: {ex.GetType().Name}: {ex.Message}\n{ex.StackTrace}");
            }
        }

        public static string GetHypothesisText(dynamic hypothesis)
        {
            var words = new List<string>();
            foreach (var item in hypothesis.items)
            {
                if (item is null)
                {
                    continue;
                }
                RetrieveWords(item, words);
            }

            return string.Join(" ", words);
        }

        public static void RetrieveWords(dynamic item, List<string> words)
        {
            if (item.type == "tag" && item.items != null)
            {
                foreach (var child in item.items)
                {
                    RetrieveWords(child, words);
                }
            }
            else if (!string.IsNullOrEmpty(item.orthography))
            {
                words.Add(item.orthography.ToString());
            }
        }
    }
}
