﻿using System.IO;
using VdkServiceClient.Constants;
using CommandLine;

namespace VoiceRecognition
{

    /// <summary>
    /// Command line options.
    /// </summary>
    public class Options
    {
        [Option('S', "scheme", HelpText = "Protocol to use for the VDK service (http or https).", Default = VdkService.Scheme)] public string Scheme { get; set; }
        [Option('h', "host", HelpText = "VDK service host.", Default = VdkService.Host)] public string Host { get; set; }
        [Option('p', "port", HelpText = "VDK service port.", Default = VdkService.Port)] public int Port { get; set; }
        [Option('f', "file", HelpText = "If given, will use a file instead of the microphone.")] public string InputFile { get; set; }
        [Option('m', "model", HelpText = "The Voice Recognition model to use.")] public string Model { get; set; }
        [Option('s', "slot", HelpText = "Format: slot:value1;value2;value3")] public string Slot { get; set; }
        [Option('r', "stop-at-first-result", HelpText = "Stop at first result.", Default = false)] public bool StopAtFirstResult { get; set; }
        [Option('l', "list", HelpText = "List available models.", Default = false)] public bool ListModels { get; set; }
        [Option('v', "verbose", HelpText = "Verbose mode (prints debug messages).", Default = false)] public bool Verbose { get; set; }

        /// <summary>
        /// Validate and skip options that don't require validation and make sure the ones that do are valid.
        /// </summary>
        /// <param name="error"> The error message. </param>
        public bool Validate(out string error)
        {
            error = null;
            if (ListModels)
            {
                error = null;
            }
            else if (string.IsNullOrWhiteSpace(Model))
            {
                error = "Model is required.";
            }
            else if (!string.IsNullOrWhiteSpace(InputFile) && !File.Exists(InputFile))
            {
                error = "Specified input audio file does not exist.";
            }
            else if (!string.IsNullOrWhiteSpace(Slot))
            {
                var slotValues = Slot.Split(';');
                if (slotValues.Length < 2)
                {
                    error = "Slot must have at least one value. Please use format: slot:value1;value2;value3";
                }
            }
            return error is null;
        }
    }
}
