using System;
using System.Collections.Generic;
using System.Json;
using System.Threading.Tasks;
using VdkServiceClient.Utils;
using VdkServiceClient.Http;


namespace VdkServiceClient.Api
{
    /// <summary>
    /// Voice Biometrics API.
    /// </summary>
    /// <remarks>
    /// Contains : get available models, get model info, users for model,
    ///            create user, delete user, enroll, authenticate,
    ///            identiy
    /// </remarks>
    /// <param name="http"> A preconfigured http client </param>
    public class VoiceBiometricsApi
    {
        public enum BiometricMode
        {
            Authentication,
            Identification
        }

        public enum BiometricType
        {
            TextDependent,
            TextIndependent
        }

        public class BiometricModelInfo
        {
            public string Name { get; set; } = "";
            public BiometricType? Type { get; set; }
            public List<string> Users { get; set; } = new List<string>();
        }

        private readonly HttpService _http;

        public VoiceBiometricsApi(HttpService http)
        {
            _http = http;
        }

        /// <summary>
        /// Retrieves the list of available models.
        /// </summary>
        /// <param name="availableModels">The list of available models to fill</param>
        public async Task<bool> GetModelsAsync(List<string> availableModels)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var result = await _http.GetAsync(Constants.Endpoints.Biometrics.GetModels);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            if (result.Data is JsonObject json)
            {
                var modelsNode = json["models"];
                if (modelsNode is JsonArray models)
                {
                    foreach (var model in models)
                    {
                        availableModels.Add(model ?? "");
                    }
                }
            }
            else
            {
                return Fail("Invalid JSON (expected object of models).");
            }
            return true;
        } // !GetModelsAsync

        /// <summary>
        /// Retrieves information about a model (model type, users)
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="model">The model to fill</param>
        public async Task<bool> GetModelInfoAsync(string modelName, BiometricModelInfo model)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var relativePath = string.Format(Constants.Endpoints.Biometrics.GetModelInfo, modelName);
            var result       = await _http.GetAsync(relativePath);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            if (result.Data is JsonObject jsonRoot)
            {
                model.Name = jsonRoot["model"] ?? "";
                model.Type = jsonRoot["model_type"] == "text_dependent" ? BiometricType.TextDependent : BiometricType.TextIndependent;

                model.Users.Clear();
                var usersNode = jsonRoot["users"];
                if (usersNode is JsonArray users)
                {
                    Logger.Debug("Available users:");
                    foreach (var user in users)
                    {
                        model.Users.Add(user ?? "");
                        Logger.Debug($"- {user}");
                    }
                }
                return true;
            }
            return Fail("Invalid JSON (expected object of model info).");
        } //!GetModelInfoAsync

        /// <summary>
        /// Retrieves the list of users for a given model.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="availableUsers">The list of available users to fill</param>
        public async Task<bool> GetModelUsersAsync(string modelName, List<string> availableUsers)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var relativePath = string.Format(Constants.Endpoints.Biometrics.GetModelUsers, modelName);
            var result       = await _http.GetAsync(relativePath);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            if (result.Data is JsonObject jsonRoot)
            {
                var usersNode = jsonRoot["users"];
                if (usersNode is JsonArray users)
                {
                    Logger.Debug("Available users:");
                    foreach (var user in users)
                    {
                        availableUsers.Add(user ?? "");
                        Logger.Debug($"- {user}");
                    }
                }

                return true;
            }
            return Fail("Invalid JSON (expected object of users).");
        } // !GetModelUsersAsync

        /// <summary>
        /// Delete a model
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        public async Task<bool> DeleteModelAsync(string modelName)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var relativePath = string.Format(Constants.Endpoints.Biometrics.DeleteModel, modelName);
            var result       = await _http.DeleteAsync(relativePath);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            return true;
        } // !DeleteModelAsync

        /// <summary>
        /// Delete an user from a model
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="user">The name of the user</param>
        public async Task<bool> DeleteModelUserAsync(string modelName, string user)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var relativePath = string.Format(Constants.Endpoints.Biometrics.DeleteModelUser, modelName, user);
            var result       = await _http.DeleteAsync(relativePath);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            return true;
        } // !DeleteUserAsync

        /// <summary>
        /// Make an enrollment request.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="modelType">The type of the model</param>
        /// <param name="user">The name of the user</param>
        public async Task<string> EnrollAsync(string modelName, BiometricType modelType, string user)
        {
            // local function
            Func<string, string> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return "";
            };

            dynamic json = new
            {
                model = modelName,
                model_type = modelType == BiometricType.TextIndependent ? "text_independent" : "text_dependent",
                user = user
            };

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Biometrics.Enroll, json);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            if (result.Data is JsonObject root)
            {
                return root["token"] ?? "";
            }
            return Fail("Invalid JSON (expected object of token).");
        } // !EnrollAsync

        /// <summary>
        /// Make an identification request.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        public async Task<string> IdentifyAsync(string modelName)
        {
            // local function
            Func<string, string> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return "";
            };

            dynamic json = new
            {
                model = modelName
            };

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Biometrics.Identify, json);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }

            if (result.Data is JsonObject root)
            {
                return root["token"] ?? "";
            }
            return Fail("Invalid JSON (expected object of token).");
        } // !IdentifyAsync

        /// <summary>
        /// Make an authentication request.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        public async Task<string> AuthenticateAsync(string modelName, string user)
        {
            // local function
            Func<string, string> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return "";
            };

            dynamic json = new
            {
                model = modelName,
                user = user
            };

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Biometrics.Authenticate, json);
            if (!result.IsSuccess)
            {
                result.displayErrors();
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");
            }
            if (result.Data is JsonObject root)
            {
                return root["token"] ?? "";
            }
            return Fail("Invalid JSON (expected object of token).");
        } // !AuthenticateAsync
    } // !VoiceBiometricsApi
}
