using System;
using System.Threading;
using System.Threading.Tasks;
using System.Net.WebSockets;
using System.Text;
using VdkServiceClient.Audio;
using VdkServiceClient.Utils;
using System.Json;

namespace VdkServiceClient.WebSocket
{
    public class WebSocketClient : IDisposable
    {
        private readonly ClientWebSocket _socket = new ClientWebSocket();
        private readonly CancellationTokenSource _cts = new CancellationTokenSource();
        private readonly TaskCompletionSource<bool> _completion = new TaskCompletionSource<bool>();
        private Task _receiveLoop;

        public event Action<JsonValue> OnMessage;
        public event Action<Exception> OnError;
        public event Action OnClose;

        public bool IsConnected => _socket.State == WebSocketState.Open;

        public async Task ConnectAsync(string url, string bearerToken = null, CancellationToken ct = default)
        {
            try
            {
                if (!string.IsNullOrWhiteSpace(bearerToken))
                    _socket.Options.SetRequestHeader("Authorization", $"Bearer {bearerToken}");

                await _socket.ConnectAsync(new Uri(url), ct);
                Logger.Debug("WebSocket connected.");

                _receiveLoop = Task.Run(() => ReceiveLoopAsync(_cts.Token));
            }
            catch (Exception ex)
            {
                Logger.Error($"❌ Failed to connect WebSocket: {ex.Message}");
                Logger.Error($"URL was: {url}");
                OnError?.Invoke(ex);
            }
        }

        public Task SendJsonAsync(JsonObject payload, CancellationToken ct = default)
        {
            return SendAsync(payload.ToString(), ct);
        }

        public async Task SendAudioAsync(byte[] pcmBytes, bool isLast = false, bool isReference = false, CancellationToken ct = default)
        {
            var base64 = Convert.ToBase64String(pcmBytes);
            dynamic msg = new
            {
                data = $"data:audio/pcm;base64,{base64}",
                last = isLast,
                is_reference = isReference
            };

            await SendAsync(msg, ct);
        }

        private Task SendAsync(dynamic msg, CancellationToken ct)
        {
            string text = "";
            if (msg is JsonValue json)
            {
                text = json.ToString();
            }
            else if (msg is string)
            {
                text = msg;
            }
            else
            {
                text = JsonExtensions.ParseDynamicObject(msg).ToString();
            }

            var buffer = new ArraySegment<byte>(Encoding.UTF8.GetBytes(text));
            return _socket.SendAsync(buffer, WebSocketMessageType.Text, true, ct);
        }

        private async Task ReceiveLoopAsync(CancellationToken ct)
        {
            var buffer = new byte[4096];
            var sb = new StringBuilder();

            try
            {
                while (!ct.IsCancellationRequested && _socket.State == WebSocketState.Open)
                {
                    var result = await _socket.ReceiveAsync(new ArraySegment<byte>(buffer), ct);

                    if (result.MessageType == WebSocketMessageType.Close)
                    {
                        await _socket.CloseAsync(WebSocketCloseStatus.NormalClosure, "Closing", ct);
                        OnClose?.Invoke();
                        _completion.TrySetResult(true);
                        return;
                    }

                    sb.Append(Encoding.UTF8.GetString(buffer, 0, result.Count));

                    if (result.EndOfMessage)
                    {
                        HandleMessage(sb.ToString());
                        sb.Clear();
                    }
                }
            }
            catch (OperationCanceledException)
            {
                // (Normal shutdown, nothing to do)
            }
            catch (Exception ex)
            {
                Logger.Error($"WebSocket receive error: {ex.Message}");
                OnError?.Invoke(ex);
            }
        }

        private void HandleMessage(string message)
        {
            try
            {
                JsonValue json = JsonValue.Parse(message);
                OnMessage?.Invoke(json);

                // Logging
                if (json is JsonObject obj && obj.ContainsKey("data"))
                {
                    obj["data"] = "base64AudioData";
                }
                Logger.Debug($"Socket message: {json}");
            }
            catch (Exception ex)
            {
                Logger.Error($"Invalid JSON from WebSocket: {ex.Message}");
                OnError?.Invoke(ex);
            }
        }

        public async void Dispose()
        {
            _cts.Cancel();

            if (_socket.State == WebSocketState.Open)
                await _socket.CloseAsync(WebSocketCloseStatus.NormalClosure, "Dispose", CancellationToken.None);

            _socket.Dispose();
            _cts.Dispose();
        }
        public Task WaitForCompletionAsync()
        {
            return _completion.Task;
        }

    }
}