from pathlib import Path
from typing import Optional
import wave

from voice_biometrics.audio.audio_utils import AudioFormat
import voice_biometrics.core.constants;

class AudioReader:
    def __init__(self,
                file_path:       str,
                format:     Optional[AudioFormat] = None):
        """
            Read a WAV file or a PCM file.
            If file_path is a WAV file, sample_rate, channels and bits_per_sample are ignored.
        """
        self.file = None
        self.is_wav = False
        self.format = format

        self.path = Path(file_path)
        ext = self.path.suffix.lower()
        if ext == ".wav":

            if format:
                raise ValueError("WAV files cannot have a format")

            self.is_wav          = True
            self.file            = wave.open(str(self.path), "rb")
            self.format          = AudioFormat(0, 0, 0)
            self.format.sample_rate     = self.file.getframerate()
            self.format.channels        = self.file.getnchannels()
            self.format.bits_per_sample = self.file.getsampwidth() * 8

        elif ext == ".pcm":
            if not format:
                self.format = voice_biometrics.core.constants.Defaults.DEFAULT_AUDIO_FORMAT

            self.file = open(self.path, "rb")

        else:
            raise ValueError(f"Unsupported extension: {ext}")

    def read_all_bytes(self) -> bytes:
        if self.is_wav:
            frames = self.file.getnframes()
            return self.file.readframes(frames)
        else:
            return self.file.read()

    def close(self) -> None:
        if self.file:
            self.file.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc, tb):
        self.close()
