from typing import List, Optional

from voice_biometrics.core.structures import VoiceBiometricsModelData
from voice_biometrics.core.constants import Uris
from voice_biometrics.core.http_result import HttpResult
from voice_biometrics.core.http_service import HttpService
from voice_biometrics.core.logger import Logger

class VoiceBiometricsClient:
    def __init__(self, http_service: HttpService):
        self._http: HttpService = http_service

    async def get_available_models(self) -> List[str]:
        available_models = []

        result : HttpResult = await self._http.get(Uris.VoiceBiometrics.GET_MODELS)
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return False

        Logger.debug(f"received : {result.raw_content}")

        for enhancer in result.data["models"]:
            available_models.append(enhancer)

        return available_models

    async def get_model_info(self, model: str) -> Optional[VoiceBiometricsModelData]:
        result : HttpResult = await self._http.get(Uris.VoiceBiometrics.GET_MODEL_INFO.format(model=model))
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        Logger.debug(f"received : {result.raw_content}")

        if "model" not in result.data:
            Logger.error(f"❌ Model not found")
            return None


        modelData = VoiceBiometricsModelData(
            name=result.data["model"],
            type="ti" if result.data["model_type"] == "text_independent" else "td",
            users=result.data["users"]
        )

        return modelData

    async def get_model_users(self, model: str) -> Optional[List[str]]:
        """
        Retrieve registered users for a model
        """
        result : HttpResult = await self._http.get(Uris.VoiceBiometrics.GET_MODEL_USERS.format(model=model))
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        Logger.debug(f"received : {result.raw_content}")

        if "users" not in result.data:
            Logger.error(f"❌ Users not found")
            return None

        return result.data["users"]

    async def delete_model(self, model: str) -> bool:
        result : HttpResult = await self._http.delete(Uris.VoiceBiometrics.DELETE_MODEL.format(model=model))
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return False

        return True

    async def delete_model_user(self, model: str, user: str) -> bool:
        """
        Delete a user from a model
        """
        result : HttpResult = await self._http.delete(Uris.VoiceBiometrics.DELETE_MODEL_USER.format(model=model, user=user))
        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return False

        return True

    async def enroll(self, model: str, biometric_type: str, user: str) -> Optional[str]:
        """
        Enroll a user in a model (creates the model if it doesn't exist)
        """
        json_body = {
            "model": model,
            "model_type": "text_independent" if biometric_type == "ti" else "text_dependent",
            "user": user
        }
        result: HttpResult = await self._http.post(Uris.VoiceBiometrics.ENROLL, json_body)

        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        if "token" not in result.data:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        return result.data["token"]

    async def identify(self, model: str) -> Optional[str]:
        """
        Identify which user from the registered ones in the model is speaking
        """
        json_body = {
            "model": model
        }
        result: HttpResult = await self._http.post(Uris.VoiceBiometrics.IDENTIFY, json_body)

        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        if "token" not in result.data:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        return result.data["token"]

    async def authenticate(self, model: str, user: str) -> Optional[str]:
        """
        Authenticate an user
        """
        json_body = {
            "model": model,
            "user": user
        }
        result: HttpResult = await self._http.post(Uris.VoiceBiometrics.AUTHENTICATE, json_body)

        if not result.is_success:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        if "token" not in result.data:
            Logger.error(f"❌ {result.status_code}: {result.error_message}")
            return None

        return result.data["token"]
