import asyncio
import traceback
from typing import Optional

from pathlib import Path

import typer

from voice_biometrics.audio.audio_utils       import stream_file, stream_microphone
from voice_biometrics.core                    import constants
from voice_biometrics.core.websocket_client   import WebSocketClient
from voice_biometrics.core.logger             import Logger
from voice_biometrics.api.vdkclient           import VdkClient

app = typer.Typer()

@app.command()
def start(
    scheme:          str  = typer.Option(constants.Defaults.DEFAULT_VDK_SCHEME, "--scheme", "-P", help="Protocol to use for the VDK service (http or https)."),
    host:            str  = typer.Option(constants.Defaults.DEFAULT_VDK_HOST,   "--host",   "-h", help="VDK service host."),
    port:            int  = typer.Option(constants.Defaults.DEFAULT_VDK_PORT,   "--port",   "-p", help="VDK service port."),
    mode:            str  = typer.Option(None,  "--mode",         "-m", help="Mode: 'auth' (authentication) or 'ident' (identification)."),
    type:            str  = typer.Option(None,  "--type",         "-t", help="Type: 'ti' (text-independent) or 'td' (text-dependent)."),
    user:            str  = typer.Option(None,  "--user",         "-u", help="Select the user that will be analyzed."),
    model:           str  = typer.Option(None,  "--model",        "-M", help="Select the model that will be used."),
    enroll:          bool = typer.Option(False, "--enroll",       "-e", help="Perform enrollment."),
    input_file:      str  = typer.Option(None,  "--file",         "-f", help="Use a file instead of microphone."),
    listModels:      bool = typer.Option(False, "--list-models",  "-l", help="List available models."),
    modelInfo:       bool = typer.Option(False, "--model-info",   "-i", help="Retrieve information about a model."),
    listUsers:       bool = typer.Option(False, "--list-users",   "-U", help="List available users for a model."),
    deleteUser:      bool = typer.Option(False, "--delete-user",  "-D", help="Delete an user from a model."),
    deleteModel:     bool = typer.Option(False, "--delete-model", "-d", help="Delete a model."),
    deleteAll:       bool = typer.Option(False, "--delete-all",   "-a", help="Delete all models."),
    verbose:         bool = typer.Option(False, "--verbose",      "-v"),
):
    code = asyncio.run(run(scheme=scheme, host=host, port=port, mode=mode, type=type, user=user, model=model,
                   enroll=enroll, input_file=input_file, listModels=listModels, modelInfo=modelInfo,
                   listUsers=listUsers, deleteUser=deleteUser, deleteModel=deleteModel,
                   deleteAll=deleteAll, verbose=verbose))
    typer.Exit(code)

async def run(
    scheme: str,
    host: str,
    port: int,
    mode: Optional[str],
    type: Optional[str],
    user: Optional[str],
    model: Optional[str],
    enroll: bool,
    input_file: Optional[str],
    listModels: bool,
    modelInfo: bool,
    listUsers: bool,
    deleteUser: bool,
    deleteModel: bool,
    deleteAll: bool,
    verbose: bool
):
    # -- validating arguments --
    if listModels or deleteAll:
        pass
    elif modelInfo:
        if not model: Logger.error("❌ Please specify a model."); return 1
    elif listUsers:
        if not model: Logger.error("❌ Please specify a model."); return 1
    elif deleteUser:
        if not model: Logger.error("❌ Please specify a model."); return 1
        if not user:  Logger.error("❌ Please specify a user."); return 1
    elif deleteModel:
        if not model: Logger.error("❌ Please specify a model."); return 1
    elif enroll:
        if not model: Logger.error("❌ Please specify a model."); return 1
        if not user:  Logger.error("❌ Please specify a user."); return 1
        if not type:  Logger.error("❌ Please specify a type."); return 1
        elif type != "ti" and type != "td":
            Logger.error("❌ Invalid type. Must be 'ti' or 'td'"); return 1
    else:
        if mode:
            if mode != "auth" and mode != "ident":
                Logger.error("❌ Invalid mode."); return 1
        else:
            Logger.error("❌ Please specify a mode."); return 1

        if not model: Logger.error("❌ Please specify a model."); return 1
        if mode == "auth" and not user:  Logger.error("❌ Please specify a user."); return 1

        if input_file and (not Path(input_file).exists() or not Path(input_file).is_file()):
            Logger.error(f"File {input_file} does not exist."); return 1

    # -- Logging configuration --
    Logger.verbose = verbose
    Logger.debug(" -- configuration -- ")
    Logger.debug(f" scheme:                 {scheme}")
    Logger.debug(f" host:                   {host}")
    Logger.debug(f" port:                   {port}")
    Logger.debug(f" mode:                   {mode}")
    Logger.debug(f" type:                   {type}")
    Logger.debug(f" user:                   {user}")
    Logger.debug(f" model:                  {model}")
    Logger.debug(f" enroll:                 {enroll}")
    Logger.debug(f" input_file:             {input_file}")
    Logger.debug(f" listModels:             {listModels}")
    Logger.debug(f" modelInfo:              {modelInfo}")
    Logger.debug(f" listUsers:              {listUsers}")
    Logger.debug(f" deleteUser:             {deleteUser}")
    Logger.debug(f" deleteModel:            {deleteModel}")
    Logger.debug(f" deleteAll:              {deleteAll}")
    Logger.debug(" ------------------------ ")

    # -----------------------------------------------------------------
    # ---------------------- running application ----------------------
    # -----------------------------------------------------------------

    client = VdkClient(scheme, host, port)
    wsClient = None
    try:
        # ------- Making sure service is reachable and ready. -------
        if not await client.health.check_healthz():
            Logger.error("❌ The VDK service is not ready.")
            return 1

        Logger.info(f"✅ VDK service is reachable.")

        # ------- Did the user requested a list of models ? -------
        if listModels:
            Logger.info("Operation : get list of available models")
            models = await client.voice_biometrics.get_available_models()
            Logger.info(f"Available models: {models}")
            return 0

        # ------- Did the user requested model info ? -------
        if modelInfo:
            Logger.info(f"Operation : get model info {model}")
            modelInfo = await client.voice_biometrics.get_model_info(model)
            Logger.info(f"Model info: {modelInfo}")
            return 0

        # ------- Did the user requested a list of users ? -------
        if listUsers:
            Logger.info(f"Operation : get list of users for model {model}")
            users = await client.voice_biometrics.get_model_users(model)
            if not users:
                Logger.error(f"❌ Could not retrieve alist of users for model {model}.")
                return 1
            Logger.info(f"Available users: {users}")
            return 0

        # ------- Did the user requested a model info ? -------
        if modelInfo:
            Logger.info(f"Operation : get model info {model}")
            modelInfo = await client.voice_biometrics.get_model_info(model)
            if modelInfo is None:
                Logger.error(f"❌ Could not find model {model}.")
                return 1
            Logger.info(f"Model info: {modelInfo}")
            return 0

        # ------- Did the user requested to delete a model ? -------
        if deleteModel:
            Logger.info(f"Operation : delete model {model}")
            if not await client.voice_biometrics.delete_model(model):
                Logger.error(f"❌ Could not delete model {model}.")
                return 1
            Logger.info(f"Model {model} deleted.")
            return 0

        # ------- Did the user requested to delete a user ? -------
        if deleteUser:
            Logger.info(f"Operation : delete user {user} from model {model}")
            if not await client.voice_biometrics.delete_model_user(model, user):
                Logger.error(f"❌ Could not delete user {user} from model {model}.")
                return 1
            Logger.info(f"User {user} deleted.")

        # ------- Did the user requested to delete all models ? -------
        if deleteAll:
            Logger.info("Operation : delete all models")
            models = await client.voice_biometrics.get_available_models()
            for model in models:
                if not await client.voice_biometrics.delete_model(model):
                    Logger.error(f"❌ Could not delete model {model}.")
                    return 1
            Logger.info("All models deleted.")
            return 0

        # -------------------- Websocket section now. --------------------
        # This is either an enrollment, identification or authentication.
        # In any case, we must retrieve a token.
        token = None
        if enroll:            token = await client.voice_biometrics.enroll(model, type, user)
        elif mode == "ident": token = await client.voice_biometrics.identify(model)
        elif mode == "auth":  token = await client.voice_biometrics.authenticate(model, user)
        else:             Logger.error(f"❌ Unknown mode {mode}."); return 1
        if token is None: Logger.error(f"❌ Could not retrieve a token."); return 1
        Logger.info(f"✅ Token: {token}")

        # ------- Starting the websocket connection -------
        wsClient = WebSocketClient()
        wsClient.on_error = lambda ex: Logger.error("❌ WebSocket error: " + str(ex))
        wsClient.on_close = lambda:    Logger.debug("WebSocket connection closed.")
        await wsClient.connect(client.ws_uri(token))
        Logger.info("✅ WebSocket connection established.")

        if enroll:            wsClient.on_message = handle_enrollment
        elif mode == "ident": wsClient.on_message = handle_identification
        elif mode == "auth":  wsClient.on_message = handle_authentication

        # ------- Either file or microphone -------
        if input_file: await stream_file(wsClient, input_file)
        else:          await stream_microphone(wsClient)
        await wsClient.wait_for_completion()
        Logger.info("✅ Done.")

    except Exception as e:
        Logger.error("❌ Application failed: " + "\n".join(traceback.format_exception(type(e), e, e.__traceback__)))
        return 1
    finally:
        if wsClient is not None:
            await wsClient.close()
        await client.close()

    return 0

def handle_enrollment(data):
    Logger.debug(f"Socket message: {data}")
    if   "event"  in data: Logger.info(f"Event: {data['event']}")
    elif "error"  in data: Logger.error(f"Error: {data['error']['code']}")
    elif "result" in data: Logger.info(f"Result: {data["result"]}")

def handle_identification(data):
    Logger.debug(f"Socket message: {data}")
    if   "event"  in data: Logger.info(f"Event: {data['event']['code_string']}")
    elif "error"  in data: Logger.error(f"Error: {data['error']['code']}")
    elif "result" in data: Logger.info(f"Result: {data['result']['id']} ({data['result']['probability']}, {data['result']['score']})")

def handle_authentication(data):
    Logger.debug(f"Socket message: {data}")
    if   "event"  in data: Logger.info(f"Event: {data['event']['code_string']}")
    elif "error"  in data: Logger.error(f"Error: {data['error']['code']}")
    elif "result" in data: Logger.info(f"Result: {data['result']['id']} ({data['result']['probability']}, {data['result']['score']})")

def main():
    app()
