import aiohttp
import json
from typing import Any, Optional

from app.core.logger import Logger
from app.core.http_result import HttpResult

class HttpService:
    def __init__(self, base_uri: str):
        self.base_uri = base_uri.rstrip("/") # making sure there is no trailing slash
        self.session : aiohttp.ClientSession = aiohttp.ClientSession(headers={"Accept": "application/json"})
        self._bearer: Optional[str] = None

    def       set_bearer(self, token: str): self._bearer = token
    async def close(self): await self.session.close()
    async def get   (self, path: str)            -> HttpResult: return await self._send("GET", path)
    async def post  (self, path: str, body: Any) -> HttpResult: return await self._send("POST", path, body)
    async def put   (self, path: str, body: Any) -> HttpResult: return await self._send("PUT", path, body)
    async def delete(self, path: str)            -> HttpResult: return await self._send("DELETE", path)

    async def _send(self, method: str, path: str, body: Any = None) -> HttpResult:
        url = f"{self.base_uri}{path}"

        Logger.info(f"[HttpService] ({method}) Request at {url}")

        headers = {}
        if self._bearer:
            headers["Authorization"] = f"Bearer {self._bearer}"

        json_body = None
        if method in ("POST", "PUT"):
            json_body = body if body is not None else {}
            Logger.debug(f"[HttpService] ({method}) Body: {json.dumps(json_body, indent=2, ensure_ascii=False)}")

        try:
            async with self.session.request(
                method=method,
                url=url,
                headers=headers,
                json=json_body
            ) as res:

                raw = await res.text()
                if not raw.strip():
                    raw = "{}"

                if not res.ok:
                    Logger.error(f"[HttpService] ({method}) {res.status}: {raw}")

                if 200 <= res.status < 300:
                    return HttpResult.from_success(res, raw)
                else:
                    return HttpResult.from_error(res, raw)

        except Exception as ex:
            return HttpResult.from_exception(ex)
