import asyncio
import time
from typing import Callable

from voice_biometrics.audio.audio_utils import AudioFormat

class AudioStreamer:

    @staticmethod
    async def stream_realtime(
        pcm_data: bytes,
        format: AudioFormat,
        callback: Callable[[bytes, bool], None],
        period_ms: int = 30
    ):
        bytes_per_sample = format.bits_per_sample // 8
        bytes_per_frame = (
            format.sample_rate
            * format.channels
            * bytes_per_sample
            * period_ms
            // 1000
        )

        start_time = time.time()
        frame_index = 0
        offset = 0
        total_len = len(pcm_data)

        while offset < total_len:
            chunk_end = min(offset + bytes_per_frame, total_len)
            chunk = pcm_data[offset:chunk_end]
            is_last = chunk_end >= total_len

            await callback(chunk, is_last)

            frame_index += 1
            next_time = start_time + (period_ms / 1000.0) * frame_index

            delay = next_time - time.time()
            if delay > 0:
                await asyncio.sleep(delay)

            offset += bytes_per_frame


    @staticmethod
    async def stream(
        pcm_data: bytes,
        callback: Callable[[bytes, bool], None],
        chunk_size: int = 400
    ):
        if chunk_size <= 0:
            raise ValueError("chunk_size must be > 0")

        total_len = len(pcm_data)
        offset = 0

        while offset < total_len:
            end = min(offset + chunk_size, total_len)
            chunk = pcm_data[offset:end]
            is_last = end >= total_len

            await callback(chunk, is_last)

            offset += chunk_size
