﻿using System.Collections.Generic;
using System.Threading.Tasks;
using System.Threading;
using VdkServiceClient.Api;
using VdkServiceClient.Audio;
using VdkServiceClient.Utils;
using VdkServiceClient.WebSocket;
using CommandLine;
using static VdkServiceClient.Api.VoiceBiometricsApi;

namespace VoiceBiometrics
{
    class Program
    {
        public static void Main(string[] args)
        {
            CommandLine.Parser.Default.ParseArguments<Options>(args)
                .WithParsed(opts =>
                {
                    if (!opts.Validate(out var err))
                    {
                        Logger.Error($"❌ {err}");
                        return;
                    }

                    Logger.SetVerbose(opts.Verbose);
                    Logger.Debug("Verbose mode enabled.");

                    RunApplication(opts).Wait();
                });
        }

        private static async Task RunApplication(Options opts)
        {
            Logger.SetVerbose(opts.Verbose);
            Logger.Debug("Verbose mode enabled.");

            var vdkClient = new VdkClient(opts.Scheme, opts.Host, opts.Port);

            // ------- Making sure service is reachable and ready. -------
            if (!await vdkClient.Core.Healthz()) 
            {
                Logger.Error("❌ VDK service is not reachable.");
                return;
            }
            Logger.Info("✅ VDK service is reachable.");

            // ------- Option : List available models -------
            if (opts.ListModels)
            {
                Logger.Debug("Listing available models...");
                var availableModels = new List<string>();
                if (await vdkClient.VoiceBiometrics.GetModelsAsync(availableModels))
                {
                    Logger.Info("Available models:");
                    foreach (var model in availableModels)
                    {
                        Logger.Info($"- {model}");
                    }
                }
                return;
            }

            // ------- Option : List available users -------
            if (opts.ListUsers)
            {
                Logger.Debug("Listing available users...");
                var availableUsers = new List<string>();
                if (await vdkClient.VoiceBiometrics.GetModelUsersAsync(opts.Model, availableUsers))
                {
                    Logger.Info($"Available users for model {opts.Model}:");
                    foreach (var user in availableUsers) Logger.Info($"- {user}");
                }
                return;
            }

            // ------- Option : Model info -------
            if (opts.ModelInfo)
            {
                Logger.Debug("Retrieving model info...");
                var model = new BiometricModelInfo();
                if (await vdkClient.VoiceBiometrics.GetModelInfoAsync(opts.Model, model))
                {
                    Logger.Info("Model info:");
                    Logger.Info($"- Name:  {model.Name}");
                    Logger.Info($"- Type:  {model.Type}");
                    Logger.Info($"- Users: [ {string.Join(", ", model.Users)} ]");
                }
                return;
            }

            // ------- Option : Delete model -------
            if (opts.DeleteModel)
            {
                Logger.Debug("Deleting model...");
                if (await vdkClient.VoiceBiometrics.DeleteModelAsync(opts.Model))
                {
                    Logger.Info("✅ Model deleted.");
                }
                return;
            }

            // ------- Option : Delete an user from a model -------
            if (opts.DeleteUser)
            {
                Logger.Debug("Deleting user...");
                if (await vdkClient.VoiceBiometrics.DeleteModelUserAsync(opts.Model, opts.User))
                {
                    Logger.Info("✅ User deleted.");
                }
                return;
            }

            // ------- Option : Delete all models -------
            if (opts.DeleteAll)
            {
                Logger.Debug("Deleting all models...");
                var availableModels = new List<string>();
                if (await vdkClient.VoiceBiometrics.GetModelsAsync(availableModels))
                {
                    foreach (var model in availableModels)
                    {
                        await vdkClient.VoiceBiometrics.DeleteModelAsync(model);
                    }
                }
                return;
            }

            // ------- Option : Enroll -------
            if (opts.Enroll)
            {
                Logger.Debug("Enrolling...");
                BiometricType? modelType = null;
                if (!string.IsNullOrEmpty(opts.Type))
                {
                    modelType = opts.Type == "ti" ? BiometricType.TextIndependent : BiometricType.TextDependent;
                }
                await PerformEnrollmentAsync(vdkClient, opts.Model, modelType, opts.User, string.IsNullOrEmpty(opts.FromFile), opts.FromFile);
                return;
            }

            // ------- Option : Identify -------
            if (opts.Mode == "ident")
            {
                Logger.Debug("Identifying...");
                await PerformIdentificationAsync(vdkClient, opts.Model, string.IsNullOrEmpty(opts.FromFile), opts.FromFile);
                return;
            }

            // ------- Option : Authenticate -------
            if (opts.Mode == "auth")
            {
                Logger.Debug("Authenticating...");
                await PerformAuthenticationAsync(vdkClient, opts.Model, opts.User, string.IsNullOrEmpty(opts.FromFile), opts.FromFile);
                return;
            }
        } // !RunAsync

        static async Task PerformEnrollmentAsync(VdkClient vdkClient, string model, BiometricType? modelType, string user, bool useMicrophone, string file)
        {
            // retrieve model info
            var modelInfo = new BiometricModelInfo();
            var retrievedModelType = modelType ?? null;
            await vdkClient.VoiceBiometrics.GetModelInfoAsync(model, modelInfo);

            // // If modeltype is null, check opts, make sure it's valid.
            if (modelInfo.Type != null)
            {
                retrievedModelType = modelInfo.Type;
            }
            else if (retrievedModelType is null)
            {
                Logger.Error("❌ Model doesn't exist. You must specify a type with -t/--type");
                return;
            }

            // Retrieve a token.
            var token = await vdkClient.VoiceBiometrics.EnrollAsync(model, retrievedModelType.Value, user);
            if (string.IsNullOrWhiteSpace(token)) 
            { 
                Logger.Error("❌ Token is null."); 
                return;
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            var cts = new CancellationTokenSource();
            var ws = new WebSocketClient();
            ws.OnClose += () => { 
                Logger.Debug("WebSocket closed."); 
                cts.Cancel();
            };
            ws.OnError += ex => { 
                Logger.Error($"WebSocket error: {ex.Message}"); 
                cts.Cancel(); 
            };
            ws.OnMessage += json => {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cts.Cancel();
                }
                else
                {
                    // Do something with the json.
                    Logger.Debug($"RESULT: {json}");
                }
            };

            await ws.ConnectAsync(vdkClient.wsUrl(token), null, cts.Token);
            if (!ws.IsConnected)
            { 
                Logger.Error("❌ WebSocket connection failed."); 
                return;
            }
            Logger.Info("✅ WebSocket connected.");

            // Now we either use microphone or a file.
            if (useMicrophone)
            {
                AudioStreamer.StreamMicrophone(ws, 16000, 1);
            }
            else
            {
                AudioStreamer.StreamFile(ws, file, 16000, 1);
            }

            cts.Token.WaitHandle.WaitOne();
        }

        static async Task PerformIdentificationAsync(VdkClient vdkClient, string model, bool useMicrophone, string file)
        {
            // Retrieve a token.
            var token = await vdkClient.VoiceBiometrics.IdentifyAsync(model);
            if (string.IsNullOrWhiteSpace(token)) 
            { 
                Logger.Error("❌ Token is null."); 
                return;
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            var cts = new CancellationTokenSource();
            var ws = new WebSocketClient();
            ws.OnClose += () => { 
                Logger.Debug("WebSocket closed."); 
                cts.Cancel(); 
            };
            ws.OnError += ex => { 
                Logger.Error($"WebSocket error: {ex.Message}"); 
                cts.Cancel(); 
            };
            ws.OnMessage += json => {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cts.Cancel();
                }
                else
                {
                    // [DEBUG] RESULT: {"result":{"id":"emmanuel","probability":1.0,"score":43.29522705078125}}
                    Logger.Debug($"RESULT: {json}");
                    var result = json["result"];
                    string id = result["id"];
                    double prob = result["probability"];
                    double score = result["score"];
                    Logger.Info($"Result : {id} (probability: {prob}, score: {score})");
                }
            };
            await ws.ConnectAsync(vdkClient.wsUrl(token), null, cts.Token);
            if (!ws.IsConnected) 
            { 
                Logger.Error("❌ WebSocket connection failed."); 
                return; 
            }
            Logger.Info("✅ WebSocket connected.");

            // Now we either use microphone or a file.
            if (useMicrophone)
            {
                AudioStreamer.StreamMicrophone(ws);
            }
            else
            {
                AudioStreamer.StreamFile(ws, file);
            }
            cts.Token.WaitHandle.WaitOne();
        }

        static async Task PerformAuthenticationAsync(VdkClient vdkClient, string model, string user, bool useMicrophone, string file)
        {
            // Retrieve a token.
            var token = await vdkClient.VoiceBiometrics.AuthenticateAsync(model, user);
            if (string.IsNullOrWhiteSpace(token)) 
            { 
                Logger.Error("❌ Token is null."); 
                return; 
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            var cts = new CancellationTokenSource();
            var ws = new WebSocketClient();
            ws.OnClose += () => { 
                Logger.Debug("WebSocket closed."); 
                cts.Cancel();
            };
            ws.OnError += ex => { 
                Logger.Error($"WebSocket error: {ex.Message}"); 
                cts.Cancel();
            };
            ws.OnMessage += json =>
            {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cts.Cancel();
                }
                else
                {
                    // [DEBUG] RESULT: {"result":{"id":"emmanuel","probability":1.0,"score":43.29522705078125}}
                    Logger.Debug($"RESULT: {json}");
                    var result = json["result"];
                    string id = result["id"];
                    double prob = result["probability"];
                    double score = result["score"];
                    Logger.Info($"Result : {id} (probability: {prob}, score: {score})");
                }
            };
            
            await ws.ConnectAsync(vdkClient.wsUrl(token), null, cts.Token);
            if (!ws.IsConnected)
            {
                Logger.Error("❌ WebSocket connection failed.");
                return;
            }
            Logger.Info("✅ WebSocket connected.");

            // Now we either use microphone or a file.
            if (useMicrophone)
            {
                AudioStreamer.StreamMicrophone(ws);
            }
            else
            {
                AudioStreamer.StreamFile(ws, file);
            }

            cts.Token.WaitHandle.WaitOne();
        }
    }
}
