﻿using System.IO;
using VdkServiceClient.Constants;
using CommandLine;

namespace VoiceBiometrics
{

    /// <summary>
    /// Command line options.
    /// </summary>
    public class Options
    {
        [Option('s', "scheme", HelpText = "Protocol to use for the VDK service (http or https).", Default = VdkService.Scheme)] public string Scheme { get; set; }
        [Option('h', "host", HelpText = "VDK service host.", Default = VdkService.Host)] public string Host { get; set; }
        [Option('p', "port", HelpText = "VDK service port.", Default = VdkService.Port)] public int Port { get; set; }
        [Option('m', "mode", HelpText = "Mode: 'auth' (authentication) or 'ident' (identification).")] public string Mode { get; set; }
        [Option('t', "type", HelpText = "Type: 'ti' (text-independent) or 'td' (text-dependent).")] public string Type { get; set; }
        [Option('u', "user", HelpText = "Select the user that will be analyzed.", Default = null)] public string User { get; set; }
        [Option('M', "model", HelpText = "Select the model that will be used.", Default = null)] public string Model { get; set; }
        [Option('c', "confidence", HelpText = "Confidence threshold (0-10). The higher, the stricter the recognizer will be.", Default = 5)] public int Confidence { get; set; }
        [Option('e', "enroll", HelpText = "Perform enrollment.", Default = false)] public bool Enroll { get; set; }
        [Option('f', "file", HelpText = "Use a file instead of microphone.", Default = null)] public string FromFile { get; set; }
        [Option('l', "list-models", HelpText = "List available models.", Default = false)] public bool ListModels { get; set; }
        [Option('i', "model-info", HelpText = "Retrieve information about a model.", Default = false)] public bool ModelInfo { get; set; }
        [Option('U', "list-users", HelpText = "List available users for a model.", Default = false)] public bool ListUsers { get; set; }
        [Option('d', "delete-model", HelpText = "Delete a model.", Default = false)] public bool DeleteModel { get; set; }
        [Option('D', "delete-user", HelpText = "Delete an user from a model.", Default = false)] public bool DeleteUser { get; set; }
        [Option('a', "delete-all", HelpText = "Delete all models.", Default = false)] public bool DeleteAll { get; set; }
        [Option('v', "verbose", HelpText = "Verbose mode (prints debug messages).", Default = false)] public bool Verbose { get; set; }

        /// <summary>
        /// Validate and skip options that don't require validation and make sure the ones that do are valid.
        /// </summary>
        /// <param name="error"> The error message. </param>
        public bool Validate(out string error)
        {
            if (Mode != null && Mode != "auth" && Mode != "ident") error = "Invalid mode (must be 'auth' or 'ident').";
            else if (Type != null && Type != "ti" && Type != "td") error = "Invalid text type (must be 'ti' or 'td').";
            else if (!string.IsNullOrEmpty(FromFile) && !File.Exists(FromFile)) error = "File must exist.";
            else if (ListModels) error = null;
            else if (DeleteAll) error = null;
            else if (ModelInfo && string.IsNullOrEmpty(Model)) error = "Missing model name.";
            else if (ListUsers && string.IsNullOrEmpty(Model)) error = "Missing model name.";
            else if (DeleteModel && string.IsNullOrEmpty(Model)) error = "Missing model name.";
            else if (DeleteUser && (string.IsNullOrEmpty(Model) || string.IsNullOrEmpty(User))) error = "Missing model or user name.";
            else if (Enroll && (string.IsNullOrEmpty(Model) || string.IsNullOrEmpty(User))) error = "Missing model or user name.";
            else if (Mode != null && Mode == "auth" && (string.IsNullOrEmpty(Model) || string.IsNullOrEmpty(User))) error = "Missing model or user name.";
            else if (Mode != null && Mode == "ident" && string.IsNullOrEmpty(Model)) error = "Missing model name.";
            else error = null;
            return error is null;
        }
    }
}
