﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Dynamic;
using System.Json;

namespace VdkServiceClient.Utils
{
    public static class JsonExtensions
    {
        public static JsonValue ParseDynamicObject(dynamic obj)
        {
            if (obj == null)
                return new JsonPrimitive((string)null);

            if (obj is JsonValue)
            {
                // Clone json value
                return JsonValue.Parse(obj.ToString());
            }

            // Primitive types
            if (obj is string)
            {
                return obj;
            }
            else if (obj is string || obj is bool || obj is int || obj is long || obj is float || obj is double || obj is decimal)
            {
                return obj;
            }

            // ExpandoObject or IDictionary<string, object>
            if (obj is IDictionary<string, object> dict)
            {
                var jsonObj = new JsonObject();
                foreach (var kvp in dict)
                {
                    jsonObj[kvp.Key] = ParseDynamicObject(kvp.Value);
                }
                return jsonObj;
            }

            // IEnumerable (arrays, lists)
            if (obj is IEnumerable enumerable)
            {
                var jsonArray = new JsonArray();
                foreach (var item in enumerable)
                {
                    jsonArray.Add(ParseDynamicObject(item));
                }
                return jsonArray;
            }

            // Fallback: use reflection for anonymous objects
            var type = obj.GetType();
            if (!type.IsPrimitive && !type.IsEnum && !type.IsValueType)
            {
                var jsonObj = new JsonObject();
                foreach (var prop in type.GetProperties())
                {
                    var value = prop.GetValue(obj);
                    jsonObj[prop.Name] = ParseDynamicObject(value);
                }
                return jsonObj;
            }

            // Primitive fallback
            return JsonValue.Parse(obj.ToString());
        }

        public static dynamic ToDynamicObject(this JsonValue jsonValue)
        {
            if (jsonValue == null || jsonValue.JsonType == JsonType.String && (string)jsonValue == null)
            {
                return null;
            }

            switch (jsonValue.JsonType)
            {
                case JsonType.String: 
                    return (string)jsonValue;

                case JsonType.Number:
                    var str = jsonValue.ToString();                    
                    return int.TryParse(str, out int i) ? i : double.Parse(str);

                case JsonType.Boolean:
                    return (bool)jsonValue;

                case JsonType.Array:
                    var list = new List<dynamic>();
                    foreach (var item in (JsonArray)jsonValue)
                    {
                        list.Add(item.ToDynamicObject());
                    }
                    return list;

                case JsonType.Object:
                    dynamic expando = new ExpandoObject();
                    var dict = (IDictionary<string, object>) expando;
                    foreach (var kvp in (JsonObject)jsonValue)
                    {
                        dict[kvp.Key] = kvp.Value.ToDynamicObject();
                    }
                    return expando;
            }
            
            // fallback
            return jsonValue.ToString();
        }
    }
}
