using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using VdkServiceClient.Utils;
using VdkServiceClient.Http;
using System.Json;


namespace VdkServiceClient.Api
{
    /// <summary>
    /// Voice Synthesis API.
    /// </summary>
    /// <remarks>
    /// Contains : get available voices, synthesize
    /// </remarks>
    /// <param name="http"> A preconfigured http client </param>
    public class VoiceSynthesisApi
    {
        private readonly HttpService _http;

        public VoiceSynthesisApi(HttpService http)
        {
            _http = http;
        }

        /// <summary>
        /// Retrieves the list of available voices.
        /// </summary>
        /// <param name="availableVoices">The list of available voices to fill</param>
        public async Task<bool> GetAvailableVoicesAsync(List<(string voiceId, int sampleRate)> availableVoices)
        {
            // local function
            Func<string, bool> Fail = (string msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var result = await _http.GetAsync(Constants.Endpoints.Synthesis.GetVoices);
            if (!result.IsSuccess)
            {
                return Fail($"{result.StatusCode} ({result.ErrorMessage}): {result.Data}");
            }

            if (result.Data is JsonObject json)
            {
                Logger.Debug("Available voices:");
                foreach (var voice in json)
                {
                    var sampleRate = voice.Value.ContainsKey("sample_rate") ? (int)voice.Value["sample_rate"] : 22050;
                    Logger.Debug($"- {voice.Key} (sample rate: {sampleRate})");
                    availableVoices.Add((voice.Key, sampleRate));
                }
                return true;
            }
            else
            {
                return Fail("Invalid JSON (expected object of voices).");
            }
        } // !GetAvailableVoicesAsync

        /// <summary>
        /// Make a synthesis request.
        /// </summary>
        /// <param name="text">The text to synthesize</param>
        /// <param name="voiceId">The voice to use</param>
        public async Task<string> SynthesizeAsync(string text, string voiceId)
        {
            dynamic json = new
            {
                text = text,
                voice_id = voiceId
            };

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Synthesis.Synthesize, json);
            if (result.IsSuccess)
            {
                if (result.Data is JsonObject root)
                {
                    return root["token"] ?? "";
                }
            }
            Logger.Error($"❌ {result.StatusCode}: {result.ErrorMessage}");
            return "";
        } // !SynthesizeAsync
    } // !VoiceSynthesisApi
}
