using System;
using System.Collections.Generic;
using System.Json;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;
using VdkServiceClient.Http;
using VdkServiceClient.Utils;

namespace VdkServiceClient.Api
{
    /// <summary>
    /// Voice Recognition API.
    /// </summary>
    /// <remarks>
    /// Contains : get available models, recognize
    /// </remarks>
    /// <param name="http"> A preconfigured http client </param>
    public class VoiceRecognitionApi
    {
        private readonly HttpService _http;

        public VoiceRecognitionApi(HttpService http)
        {
            _http = http;
        }

        /// <summary>
        /// Retrieves the list of available models.
        /// </summary>
        /// <param name="availableModels">The list of available models to fill</param>
        public async Task<bool> GetAvailableModelsAsync(List<(string modelName, List<string> slots)> availableModels)
        {
            // local function
            Func<string, bool> Fail = (msg) => {
                Logger.Error("❌ " + msg);
                return false;
            };

            var result = await _http.GetAsync(Constants.Endpoints.Recognition.GetModels);
            if (!result.IsSuccess)
                return Fail($"{result.StatusCode}: {result.ErrorMessage}");

            if (result.Data is JsonObject json)
            {

                Logger.Debug("Available models:");
                availableModels.Clear();
                foreach (var model in json)
                {
                    Logger.Debug($"- {model.Key}");

                    var listSlots = new List<string>();
                    if (model.Value.ContainsKey("slots"))
                    {
                        var slots = model.Value?["slots"] as JsonArray;
                        if (slots != null)
                        {
                            foreach (var slot in slots)
                            {
                                listSlots.Add(slot ?? "");
                            }
                        }
                        Logger.Debug($"  Slots: {string.Join(", ", listSlots)}");
                    }
                    availableModels.Add((model.Key, listSlots));
                }

                return true;
            }
            return Fail("Invalid JSON (expected object of models).");
        } // ! CheckAndGetAvailableModelsAsync

        /// <summary>
        /// Make a recognition request.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="slot">The name of the slot (optional, format name:value1;value2;value3)</param>
        /// <param name="stopAtFirstResult">Whether to stop at the first result</param>
        /// <param name="user"> Optional user (UserWord) </param>
        public async Task<string> RecognizeAsync(string modelName,
                                                 string slot,
                                                 bool stopAtFirstResult,
                                                 string user = null)
        {
            var json = new JsonObject
            {
                ["stop_at_first_result"] = stopAtFirstResult,
                ["models"] = new JsonObject()
                {
                    [modelName] = new JsonObject()
                }
            };

            if (!string.IsNullOrWhiteSpace(slot))
            {
                // Splitting slotName and slotValues
                Func<string, (string name, string[] values)> Parse = (s) =>
                {
                    var parts = s.Split(new[] { ':' }, 2);
                    var name = parts[0];
                    var vals = parts.Length > 1
                        ? parts[1].Split(new[] { ';' }, StringSplitOptions.RemoveEmptyEntries)
                        : Array.Empty<string>();
                    return (name, vals);
                };

                // Adding slot and its values.
                var (slotName, slotValues) = Parse(slot);
                var values = new JsonArray(slotValues.Select(v => (JsonValue)v).ToArray());
                ((JsonObject)json["models"])[modelName] = new JsonObject
                {
                    ["slots"] = new JsonObject
                    {
                        [slotName] = new JsonObject
                        {
                            ["values"] = values
                        }
                    },
                    ["user"] = user
                };
            }

            Logger.Debug($"JSON: {json}");

            // Making the request, we retrieve a token.
            var result = await _http.PostAsync(Constants.Endpoints.Recognition.Recognize, json);

            if (result.IsSuccess)
            {
                if (result.Data is JsonValue root)
                {
                    return root["token"] ?? "";
                }
            }
            Logger.Error($"❌ {result.StatusCode}: {result.ErrorMessage}");
            Logger.Error($"Response : {result.RawContent}");
            return "";
        } // !RecognizeAsync

        /// <summary>
        /// Retrieves the list of user words.
        /// </summary>
        /// <param name="userWords">The list of user words to fill</param>
        public async Task<bool> GetUserWordsAsync(List<JsonObject> userWords)
        {
            if (userWords == null)
                throw new ArgumentNullException(nameof(userWords));

            var result = await _http.GetAsync(Constants.Endpoints.Recognition.GetUserWords);
            if (!result.IsSuccess)
                return false;

            var json = result.Data as JsonObject;
            if (json == null)
                return false;

            if (!json.ContainsKey("userwords"))
                return false;

            var userWordsArray = json["userwords"] as JsonArray;
            if (userWordsArray == null)
                return false;

            userWords.Clear();

            foreach (var item in userWordsArray)
            {
                var obj = item as JsonObject;
                if (obj != null)
                {
                    userWords.Add(obj);
                }
            }

            return true;
        }

        /// <summary>
        /// Make an enrollment request.
        /// </summary>
        /// <param name="modelName">The name of the model</param>
        /// <param name="user">The name of the user</param>
        /// <param name="word">The word to enroll</param>
        /// <param name="stopAtFirstResult">Whether to stop at the first result</param>
        public async Task<string> EnrollAsync(string modelName, string user, string word, bool stopAtFirstResult)
        {
            var json = new JsonObject
            {
                ["model"] = modelName,
                ["user"] = user,
                ["word"] = word,
                ["stop_at_first_result"] = stopAtFirstResult
            };

            var result = await _http.PostAsync(Constants.Endpoints.Recognition.Enroll, json);
            if (result.IsSuccess)
            {
                if (result.Data is JsonValue root)
                {
                    return root["token"] ?? "";
                }
            }
            Logger.Error($"❌ {result.StatusCode}: {result.ErrorMessage}");
            Logger.Error($"Response : {result.RawContent}");
            return "";
        }

        /// <summary>
        /// Delete an userword user.
        /// </summary>
        /// <param name="user">The name of the user</param>
        /// <param name="model">Optional model name to limit extent of deletion</param>
        public async Task<bool> DeleteUserWordsUserAsync(string user, string model = null)
        {
            var json = new JsonObject
            {
                ["model"] = model
            };

            var result = await _http.DeleteAsync(string.Format(Constants.Endpoints.Recognition.DeleteUserWordsUser, user), json);
            return result.IsSuccess;
        }

        /// <summary>
        /// Delete all user words.
        /// </summary>
        /// <param name="model">Optional model name to limit extent of deletion</param>
        public async Task<bool> DeleteUserWordsAsync(string model = null)
        {
            var json = new JsonObject
            {
                ["model"] = model
            };

            var result = await _http.DeleteAsync(Constants.Endpoints.Recognition.DeleteUserWords, json);
            return result.IsSuccess;
        }
    } // !VoiceRecognitionApi
}
