using System.Linq;
using System.Net.Http.Headers;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System;
using VdkServiceClient.Utils;
using System.Json;
using System.Collections.Generic;
using System.Collections;

namespace VdkServiceClient.Http
{
    public class HttpService
    {
        private readonly HttpClient _client;

        public HttpService(Uri baseUri)
        {
            _client = new HttpClient
            {
                BaseAddress = baseUri
            };
            if (!_client.DefaultRequestHeaders.Accept.Any(h => h.MediaType == Constants.Http.JsonContentType))
                _client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue(Constants.Http.JsonContentType));
        }

        public void SetBearer(string token)
        {
            _client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", token);
        }

        public Task<HttpResult> GetAsync(string path, CancellationToken ct = default)
        {
            return SendAsync(HttpMethod.Get, path, null, ct);
        }

        public Task<HttpResult> PostAsync(string path, object body, CancellationToken ct = default)
        {
            return SendAsync(HttpMethod.Post, path, body, ct);
        }

        public Task<HttpResult> PutAsync(string path, object body, CancellationToken ct = default)
        {
            return SendAsync(HttpMethod.Put, path, body, ct);
        }

        public Task<HttpResult> DeleteAsync(string path, object body = null, CancellationToken ct = default)
        {
            return SendAsync(HttpMethod.Delete, path, null, ct);
        }

        private async Task<HttpResult> SendAsync(HttpMethod method, string path, object body, CancellationToken ct)
        {
            try
            {
                Logger.Info($"[HttpService] ({method}) Request at {_client.BaseAddress}{path}");
                var req = new HttpRequestMessage(method, _client.BaseAddress + path);
                req.Content = body is null  ? null : new StringContent(serialize(body), Encoding.UTF8, Constants.Http.JsonContentType);

                var res = await _client.SendAsync(req, ct);
                var raw = await res.Content.ReadAsStringAsync();
                if (string.IsNullOrWhiteSpace(raw))
                {
                    raw = "{}";
                }

                return res.IsSuccessStatusCode
                    ? HttpResult.FromSuccess(res, raw)
                    : HttpResult.FromError(res, raw);
            }
            catch (Exception ex)
            {
                return HttpResult.FromException(ex);
            }
        }

        private static string serialize(object o)
        {
            JsonValue j = JsonExtensions.ParseDynamicObject(o);
            return j.ToString();
        }

    }
}
