﻿using System.Collections.Generic;
using System.Threading.Tasks;
using System.Threading;
using VdkServiceClient.Api;
using VdkServiceClient.Audio;
using VdkServiceClient.Utils;
using VdkServiceClient.WebSocket;
using CommandLine;
using System.Json;
using System;

namespace SpeechEnhancement
{
    class Program
    {
        const int SampleRate = 16000;
        const int ChannelCount = 1;

        public static void Main(string[] args)
        {
            CommandLine.Parser.Default.ParseArguments<Options>(args)
                .WithParsed(opts =>
                {
                    if (!opts.Validate(out var err))
                    {
                        Logger.Error($"❌ {err}");
                        return;
                    }

                    Logger.SetVerbose(opts.Verbose);
                    Logger.Debug("Verbose mode enabled.");

                    RunApplication(opts).Wait();
                });
        }

        private static async Task RunApplication(Options opts)
        {
            Logger.SetVerbose(opts.Verbose);
            Logger.Debug("Verbose mode enabled.");

            var vdkClient = new VdkClient(opts.Scheme, opts.Host, opts.Port);

            // ------- Making sure service is reachable and ready. -------
            if (!await vdkClient.Core.Healthz())
            {
                Logger.Error("❌ VDK service is not reachable.");
                return;
            }
            Logger.Info("✅ VDK service is reachable.");

            // ------- Retrieve a list of availables enhancers ------
            var availableEnhancers = new List<string>();
            if (!await vdkClient.SpeechEnhancement.GetAvailableEnhancersAsync(availableEnhancers) || availableEnhancers.Count == 0)
            {
                Logger.Error("❌ Could not retrieve available enhancers.");
                return;
            }

            // Did the user requested to list available enhancers?
            if (opts.ListEnhancers)
            {
                Logger.Info("Available enhancers:");
                foreach (var enhancer in availableEnhancers)
                {
                    Logger.Info($"- {enhancer}");
                }
                return;
            }

            // ------- Asking for an enhancement (retrieving a token) -------
            if (!availableEnhancers.Contains(opts.Enhancer))
            {
                Logger.Error($"❌ Enhancer '{opts.Enhancer}' not found.");
                return;
            }
            var token = await vdkClient.SpeechEnhancement.EnhanceAsync(opts.Enhancer);
            if (string.IsNullOrWhiteSpace(token))
            {
                Logger.Error("❌ Token is null.");
                return;
            }
            Logger.Info($"✅ {token}");

            // ------- Establishing the WebSocket connection -------
            var cts = new CancellationTokenSource();

            // Check extension for wav or pcm
            var outFileExt = opts.Output.EndsWith(".wav") ? "wav" : "pcm";
            var outFile = outFileExt == "wav"
                ? new AudioFileWriter(opts.Output, SampleRate, ChannelCount)
                : new AudioFileWriter(opts.Output);

            var ws = new WebSocketClient();
            ws.OnClose += () => {
                Logger.Debug("WebSocket closed.");
                cts.Cancel();
            };
            ws.OnError += ex => {
                Logger.Error($"WebSocket error: {ex.Message}");
                cts.Cancel();
            };
            ws.OnMessage += json => {
                if (json.ContainsKey("event"))
                {
                    Logger.Info($"Event: {json}");
                }
                else if (json.ContainsKey("error"))
                {
                    Logger.Error($"Error: {json["error"]["message"]}");
                }
                else if (json.ContainsKey("type") && json["type"] == "EndOfConnection")
                {
                    Logger.Info("✅ EndOfConnection received."); cts.Cancel();
                }
                else
                {
                    HandleEnhanceResult(json, outFile, cts); // see below for HandleSynthesizeResult
                }
            };
            await ws.ConnectAsync(vdkClient.wsUrl(token), null, cts.Token);
            if (!ws.IsConnected)
            {
                Logger.Error("❌ WebSocket connection failed.");
                return;
            }
            Logger.Info("✅ WebSocket connected.");

            // ------- Sending audio data -------

            Thread referenceStreamer = new Thread(()=>{
                // Do the options contain a reference audio ?
                if (!string.IsNullOrWhiteSpace(opts.Reference))
                {
                    // We'll stream the reference audio in parallel.
                    // The important thing about aec/barge-in is that the service receive the first paquet of reference before input.
                    AudioStreamer.StreamFile(ws, opts.Reference, SampleRate, 1, isReference: true);
                }
            });
            referenceStreamer.Start();

            Thread mainStreamer = new Thread(() =>
            {
                AudioStreamer.StreamFile(ws, opts.Input, SampleRate, ChannelCount, isReference: false);
            });
            mainStreamer.Start();


            mainStreamer.Join();
            referenceStreamer.Join();
            cts.Token.WaitHandle.WaitOne();
            ws.Dispose();
            outFile.Dispose();
            Logger.Info("✅ Done.");
        } // !RunAsync

        static void HandleEnhanceResult(JsonValue result, AudioFileWriter outFile, CancellationTokenSource cts)
        {
            try
            {
                // Audio data must contains 'data' and 'last' fields
                if (!result.ContainsKey("data") && !result.ContainsKey("last"))
                {
                    return;
                }

                var buffer = AudioBuffer.FromAudioPacket(result, SampleRate, ChannelCount);
                outFile?.WriteChunk(buffer.Data); // Writing audio in output.pcm

                if (result["last"])
                {
                    Logger.Info("✅ Received last chunk.");
                    cts.Cancel();
                }
            }
            catch (Exception ex)
            {
                Logger.Error($"Failed to parse ASR result: {ex.GetType().Name}: {ex.Message}\n{ex.StackTrace}");
            }
        }
    }
}
