﻿using System;
using System.Threading;
using System.Threading.Tasks;
using VdkServiceClient.Utils;
using VdkServiceClient.WebSocket;

namespace VdkServiceClient.Audio
{
    public static class AudioStreamer
    {
        /// <summary>
        /// A cancellable operation that streams the microphone into the WebSocket
        /// </summary>
        public static async Task StreamMicrophone(WebSocketClient ws, int sampleRate = 16000, int channelCount = 1, bool isReference = false)
        {
            Logger.Info("Using microphone");
            var capture = new AudioRecorder(sampleRate, channelCount);
            capture.OnAudioData += audioBuffer =>
                _ = ws.SendAudioAsync(audioBuffer.Data, false, isReference).ContinueWith(task =>
                {
                    if (task.IsFaulted)
                    {
                        Logger.Error($"Send failed: {task.Exception?.GetBaseException().Message}");
                        capture.Stop();
                    }
                });
            Console.CancelKeyPress += (s, e) => {
                e.Cancel = true;
                capture.Stop();
                Logger.Info("Capture stopped.");
                _ = ws.SendAudioAsync(Array.Empty<byte>(), true, isReference).ContinueWith(task => {
                    if (task.IsFaulted)
                    {
                        Logger.Error($"Send failed: {task.Exception?.GetBaseException().Message}");
                    }
                });
            };

            capture.Start();
            Logger.Info("Capture started.");
            Logger.Info("Press Ctrl+C to stop the capture.");
            await ws.WaitForCompletionAsync();
            Console.CancelKeyPress -= (s, e) => {};
        }

        /// <summary>
        /// An operation that streams a file into the WebSocket
        /// Assumes the file exists.
        /// </summary>
        public static void StreamFile(WebSocketClient ws, string file, int sampleRate = 16000, int channels = 1, bool isReference = false)
        {
            Logger.Info("Using file");
            var reader = new AudioFileReader(file, sampleRate, channels);
            var data = reader.ReadAllBytes();
            Stream(data, reader.SampleRate, reader.Channels, (chunk, isLast) =>
            {
                ws.SendAudioAsync(chunk, isLast, isReference).ContinueWith(task =>
                {
                    if (task.IsFaulted)
                    {
                        Logger.Error($"Send failed: {task.Exception?.GetBaseException().Message}");
                    }
                });

                if (isLast)
                {
                    Logger.Info("✅ Last chunk sent.");
                }
            });
            Logger.Info("✅ File sent.");
        }


        private static void Stream(byte[] pcmData, int sampleRate, int channelCount, Action<byte[], bool> callback, int periodMs = 30)
        {
            const int bytesPerSample = 2;
            int bytesPerFrame = sampleRate * channelCount * bytesPerSample * periodMs / 1000;


            var chunks = AudioBuffer.SplitByteArray(pcmData, bytesPerFrame);
            var start = DateTime.UtcNow;
            for (int i = 0; i < chunks.Count; i++)
            {
                callback(chunks[i], i == chunks.Count - 1);

                var nextTime = start.AddMilliseconds(periodMs * i);
                var delay = nextTime - DateTime.UtcNow;
                if (delay.TotalMilliseconds > 0)
                    Thread.Sleep(delay);
            }
        }
    }
}
