using System;
using System.Threading;
using NAudio.Wave;
using VdkServiceClient.Utils;

namespace VdkServiceClient.Audio
{
    public class AudioPlayer : IDisposable
    {
        private readonly WaveOutEvent _waveOut;
        private readonly BufferedWaveProvider _waveProvider;
        public int SampleRate { get { return _waveProvider.WaveFormat.SampleRate; } }
        public int Channels { get { return _waveProvider.WaveFormat.Channels; } }

        public AudioPlayer(int sampleRate = 22050, int channels = 1)
        {
            _waveOut = new WaveOutEvent();
            _waveProvider = new BufferedWaveProvider(new WaveFormat(sampleRate, 16, channels));
            _waveProvider.BufferDuration = TimeSpan.FromSeconds(600); // Max Buffer Duration 10 Minutes
            _waveOut.DesiredLatency = 100; // In Millis
            _waveOut.Init(_waveProvider);
        }

        public void AddAudioBuffer(AudioBuffer buffer)
        {
            if (buffer.Channels   != _waveProvider.WaveFormat.Channels ||
                buffer.SampleRate != _waveProvider.WaveFormat.SampleRate)
            {
                throw new Exception("Invalid audio buffer format.");
            }

            _waveProvider.AddSamples(buffer.Data, 0, buffer.Data.Length);
            if (_waveOut.PlaybackState != PlaybackState.Playing)
            {
                _waveOut.Play();
            }
        }

        public void Stop()
        {
            _waveOut.Stop();
            _waveProvider.ClearBuffer();
        }

        public void WaitForCompletion()
        {
            SpinWait.SpinUntil(() => _waveProvider.BufferedBytes <= 0);
        }

        public void Dispose()
        {
            _waveOut.Stop();
            _waveOut.Dispose();
        }
    }
}